;FILE: FFT-SSB.ASM, Last edition: 28-FEB-1996
;A quick hack of the FFT-CW.ASM to make an FFT display for SSB
;(c) 1995 Pawel Jalocha, SP9VRC,
;e-mail: jalocha@chopin.ifj.edu.pl, sp9vrc@gw.sp9kby.ampr.org
;
;This software is not to be used for purposes other than amateur radio
;without a written permision of the author.
;Usage for profit or in commercial/military products is explicitly prohibited.
;

	page 132
	nolist
	include 'leonid'
	list

	scsjmp short    ;force short jumps for .if/.endi, .while/.endw, etc.

        title 'FFT display and filter for SSB by SP9VRC'

;*************************************************************************
;compile-time constants

NewBaudRate     equ  57600      ;change the SCI baud rate to this value
                                ;for faster data transfer

SampleFreq equ 8000.0           ;sampling frequency [Hz]

AliasFilterLen equ 64           ;the length of the anti-alias FIR filter
                                ;Longer length => the filter's edges are
                                ;sharper and we can do larger decimation
                                ;for same practical bandwidth.

AliasCenter  equ 2000.0         ;The center and width at -6 dB and in Hz
AliasWidth   equ 3800.0         ;of the anti-alias filter.
                                ;Here we decide about the selectivity
                                ;of the filter.

DecimateRatio equ 2             ;decimation ratio after the alias filter
                                ;the maximum bandwidth of the anti-alias filter:
                                ;sampled bandwidth = SampleFreq/DecimateRadio

                                ;64-tap filter, with center at 800 Hz
                                ;and 500 Hz bandwidth shows the bandwidth
                                ;of 1600 Hz at the -110..120 dB level
                                ;thus the sampled bandwidth can safely
                                ;be 8000/5 = 1600 Hz.
                                ;For a 128-tap filter we could decimate by 8.

WindowLen       equ 256         ;sliding-window length (= FFT length)
WindowLenLog    equ @cvi(@log(WindowLen)/@log(2)+0.5)   ;2-base logarythm
                                ;256 bins imply 8000/5/256 = 6.25 Hz/bin
                                ;I think 128 is optimal for average CW
                                ;but 64 seems better for the display

FFTprocLen      equ WindowLen   ;how many FFT bins we process
FFTprocStart    equ WindowLen/2-FFTprocLen/2 ;at which FFT bin we start processing
FFTprocLenLog   equ WindowLenLog ;all the other bins we discard at receive
                                ;and for output we set them to zero.
                                ;This is an additional stage of filtering
                                ;which decides how wide or narrow
                                ;the filter will be.

                                ;64 bins make 1/4 of the total 256 thus here
                                ;we reduce the bandwidth to 1600/4 = 400 Hz.

FinerShaping    equ 0           ;0 => sin(x) window
                                ;1 => sin(x)^2 window but twice as many
                                ;     FFTs in same time thus more CPU cycles
        if FinerShaping
FFTstep    equ WindowLen/4
FFTstepLog equ WindowLenLog-2
        else
FFTstep    equ WindowLen/2
FFTstepLog equ WindowLenLog-1
        endif

TxAtten         equ 0.0         ;[dB] attenuate the output CODEC's signal.

RxGain          equ 0.0         ;[dB] Rx CODEC's input gain
                                ;set this between 0 and 22.5 dB depending on
				;the audio level from your tranceiver
RxAGC           equ 1           ;enable the automatic gain adjustment
				;for the CODEC's audio input.
				;ideally you should avoid the AGC and set
				;RxGain for best audio level match.
RxAGC_MSfollow  equ 1.0/16.0    ;update weight per FFTstep
				;for the Mean Square tracking
RxAGChold       equ 320 ;[FFTstep]  AGC peak hold time
RxAGCfall       equ 32  ;[FFTstep]  AGC falling period/1.5 dB step
RxAGCminMS      equ 0.008       ;min. and max. tolerable audio Mean Square level
RxAGCmaxMS      equ 0.04
RxAGCmaxPeak    equ 0.75        ;max. allowed peak audio level
RxAGCsensePeak  equ 0           ;take into acount the peak level
				;when deciding about the gain changes
				;taking the peak into account makes the AGC
				;more "jumpy" which is not good in my opinion.

RightChannel    equ 0           ;Use the right CODEC's channel not the left
                                ;for input. For output both channels are used
                                ;to produce the pseudo-stereo effect.

PowerPeakHold   equ 100         ;the time [in FFTstep] to hold the power peak
                                ;level of the spectrum. This level is used
                                ;as a reference for suppressing the noise.
PowerPeakFall   equ 10          ;the time [in FFTstep] between succesive drops
                                ;of the power peak after the PowerPeakHold
                                ;has expired.
NoiseFloorExclude equ WindowLen/8 ;that many strongest powers we exclude
                                ;before taking the average power
                                ;to extimate the noise power.
NoiseFloorFollow equ 512/WindowLen ;the weight to integrate the noise floor
Threshold       equ 4           ;threshold for the cut like in the FFT-CUT
                                ;4 seems to be optimal
ParabolicSmooth equ 2           ;0,1 or 2; like for the FFT-CUT

RxPipeLen equ 1<<NoiseFloorFollow ;FFT processing pipe length


SendSigToThres  equ 1           ;send the signal-to-threshold values
                                ;on the serial port for external
                                ;FFT flow display. SPY must be OFF !

BufLen   equ      512              ;sample buffer length
				   ;with some safety margin
                                   ;this should be at least 2*AliasFilterLen

SPY      equ 0  ; SPY=0 => no debug (ALL SPY_ sub-options MUST be set to 0 !)
                ; SPY=1 => selected debug code included

SPY_RxIQ        equ 0   ;I/Q after the Hilbert transformer and Doppler corrector
SPY_RxWindow    equ 0
SPY_RxFFT       equ 0   ;FFT values on the early Rx stage
SPY_PeakMS      equ 0   ;Signal's peak and MS seen by the AGC
SPY_CodecGain   equ 0   ;CODEC's gain being changed by the AGC
SPY_RxCarIQ     equ 0   ;I/Q for the carriers after the FFT and phase corr.
SPY_RxCodecPtr  equ 0   ;the Rx CODEC pointer
SPY_RxCarFreq   equ 0   ;the Doppler correction

MonitorRxAGC    equ 0           ;monitor the gain changes being done by the AGC
				;with the UP and DOWN LEDs
				;This option conflicts with RxCtrlUpDown
				;when you use the left channel.

;*************************************************************************
;Some simple macros

      if RightChannel
PTT     macro mode      ;PTT line: clr/set/chg
	b\mode #4,X:$FFE4
	endm
      else
PTT     macro mode      ;PTT line: clr/set/chg
	b\mode #0,X:$FFE4
;      b\mode #3,X:$FFE4       ;*** DEBUG ***
	endm
      endif

      if RightChannel
PushUp  macro mode      ;Push UP line of the TRX
	b\mode #5,X:$FFE4
	endm
      else
PushUp  macro mode
	b\mode #1,X:$FFE4
	endm
      endif

      if RightChannel
PushDown macro mode      ;Push DOWN line of the TRX
	b\mode #6,X:$FFE4
	endm
      else
PushDown macro mode
	b\mode #2,X:$FFE4
	endm
      endif

UpLED   macro mode      ;UP line (a red LED connected)
	b\mode #1,X:$FFE4
	endm

DownLED macro mode      ;DOWN line (a red LED connected)
	b\mode #2,X:$FFE4
	endm

YellowLED macro mode    ;CAT line (a yellow LED connected)
	b\mode #3,X:$FFE4
	endm

RedLED  macro mode      ; Red LED clr/set/chg
	b\mode #13,X:$FFE4
	endm

LongDecayAver macro Input,Aver,Scale
	      sub Aver,Input
	      rep Scale
		asr Input
	      add Input,Aver
	      endm

;*************************************************************************
;The actuall code (internal/external program RAM)

	LOMEM P:$0000
	HIMEM P:$1FFF

	org     p:user_code

	jmp <Initialize

DoFFT   ;routine to perform a complex FFT (and Inverse FFT too)
	;copied from Motorola's Dr. BuB.
	;alters a,b,x,y, rnm0, rnm1, n2, rnm4, rnm5, rnm6
	;Uses 6 locations on System Stack

	 move    #WindowLen/2,n0   ;initialize butterflies per group
	 move    #1,n2             ;initialize groups per pass
	 move    #WindowLen/4,n6   ;initialize C pointer offset
	 move    #-1,m0            ;initialize A and B address modifiers
	 move    m0,m1             ;for linear addressing
	 move    m0,m4
	 move    m0,m5
	 move    #0,m6             ;initialize C address modifier for
				   ;reverse carry (bit-reversed) addressing
;
; Perform all FFT passes with triple nested DO loop
;
	 do      #WindowLenLog,_end_pass
	 move    #FFTbuff,r0     ;initialize A input pointer
	 move    r0,r4           ;initialize A output pointer
	 lua     (r0)+n0,r1      ;initialize B input pointer
	 move    #FFTcoef,r6     ;initialize C input pointer
	 lua     (r1)-,r5        ;initialize B output pointer
	 move    n0,n1           ;initialize pointer offsets
	 move    n0,n4
	 move    n0,n5

	 do      n2,_end_grp
	 move    x:(r1),x1  y:(r6),y0        ;lookup -sine and 
					     ; -cosine values
	 move    x:(r5),a   y:(r0),b         ;preload data
	 move    x:(r6)+n6,x0                ;update C pointer

	 do      n0,_end_bfy
	 mac     x1,y0,b    y:(r1)+,y1       ;Radix 2 DIT
					     ;butterfly kernel
	 macr    -x0,y1,b   a,x:(r5)+    y:(r0),a
	 subl    b,a        x:(r0),b     b,y:(r4)
	 mac     -x1,x0,b   x:(r0)+,a  a,y:(r5)
	 macr    -y1,y0,b   x:(r1),x1
	 subl    b,a        b,x:(r4)+  y:(r0),b
_end_bfy
	 move    a,x:(r5)+n5    y:(r1)+n1,y1   ;update A and B pointers
	 move    x:(r0)+n0,x1   y:(r4)+n4,y1
_end_grp
	 move    n0,b1
	 lsr     b   n2,a1     ;divide butterflies per group by two
	 lsl     a   b1,n0     ;multiply groups per pass by two
	 move    a1,n2
_end_pass
	rts

ProcessLoop     ;main loop to process CODEC's samples

RxProcess
	move X:<WindowInpTapPtr,r1 ;r1 to address the input window tap
	move #<WindowLen-1,m1
        move #>DecimateRatio*FFTstep,a  ;a = how many samples we should have before
	move X:<RxCodecPtr,r2     ;making the new window
	jsr <WaitSampleBlock      ;wait for that many samples to come
	move #BufLen*4-1,m2       ;set m2 so r2 wraps correctly
	move #<4,n2
	if SPY_RxCodecPtr
	  move r2,a
	  rep #8
	    asl a
	  jsr <SpyA
	endif
	move #<AliasFilterLen-1,m4
      if RightChannel
	move (r2)+
      endif
      if RxAGC
        move r2,r0              ;save r2
        .loop #DecimateRatio    ;update mean input audio power
          clr a X:(r2)+n2,x0
          .loop #FFTstep-1
            mac x0,x0,a X:(r2)+n2,x0
          .endl
          mac x0,x0,a X:<RxAudioMS,x1
          rep #FFTstepLog
            asr a
          move a,x0 #RxAGC_MSfollow,y0
          mpy x0,y0,a #1.0-RxAGC_MSfollow,y0
          macr x1,y0,a
          move a,X:<RxAudioMS
        .endl
        move r0,r2              ;restore r2
       if RxAGCsensePeak
        move X:<RxAudioPeak,a   ;update the peak value of the input signal
	move r2,r0              ;save r2
        .loop #DecimateRatio*FFTstep  ;look through the new samples
	  move X:(r2)+n2,x0     ;and find the largest one
	  cmpm x0,a
	  tlt x0,a
	.endl
	abs a r0,r2              ;restore r2
	move a,X:<RxAudioPeak
;        move X:<RxAudioSatFract,a       ;measure the percentage of time
;        move #RxSatLevel,y0             ;the input audio goes above
;        move #RxSatTrackWeight,x0       ;the saturation level
;        move r2,r0              ;save r2
;        .loop #DecimateRatio*SymbolLen/2
;          move a,x1                     ;a = a - weight*a = a*(1.0-weight)
;          mpyr -x0,x1,a X:(r2)+n2,b
;          cmpm y0,b #<0,b               ;if |x|>=satur.
;          tge x0,b                      ;then a += weight
;          add b,a                       ;else a += 0
;        .endl
;        move r0,r2
;        move a,X:<RxAudioSatFract
       endif
      endif
        .loop #FFTstep            ;filter, decimate and fill the window's tap
	  move #4*(AliasFilterLen-DecimateRatio),n2
	  move #AliasFilterInpI,r4 ;r4 to address the input filter shape
	  move (r2)-n2            ;move back in the CODEC input buffer
	  move #<4,n2             ;n2=4 for easier input adressing
	  move r2,x1              ;save r2 (where we start the filter tap)
	  clr a X:(r2)+n2,x0 Y:(r4)+,y0
	  .loop #AliasFilterLen-1
	    mac x0,y0,a X:(r2)+n2,x0  Y:(r4)+,y0
	  .endl
	  macr x0,y0,a #AliasFilterInpQ,r4
	  move x1,r2      ;restore the tap start
	  nop
	  clr b X:(r2)+n2,x0 Y:(r4)+,y0
	  .loop #AliasFilterLen-1
	    mac -x0,y0,b X:(r2)+n2,x0  Y:(r4)+,y0
	  .endl
	  macr -x0,y0,b a,X:(r1) ;a,b = decimated sample (I/Q), restore r2
          move b,Y:(r1)+         ;put new sample into the tap
;          move X:<RxCarPhase,x0  ;get the receiver NCO's carrier phase
;          move X:<RxCarFreq,a    ;advance the phase
;          add x0,a
;          move a1,X:<RxCarPhase
;          jsr <IQ               ;compute I and Q (modifies a,b,x,y,r0,m0,n0)
;          move b,y0
;          move X:(r1),x1 a,y1
;          mpy x1,y1,a Y:(r1),x0 ;mix the input samples with the NCO
;          macr -x0,y0,a
;          mpy x1,y0,b
;          macr x0,y1,b a,X:(r1)
;          move b,Y:(r1)+
	if SPY_RxIQ
	  jsr <SpySync
	  .if <cc>
	    jsr <SpyA
	    tfr b,a
	    jsr <SpyA
	    nop
	  .endi
	endif
	.endl
      if RightChannel
	move (r2)-
      endif
	move r2,X:<RxCodecPtr     ;save the CODEC buffer pointer

RxSlide         ;apply window, copy data to FFTbuff
        move #RxWindow,r5
        move #WindowLen-1,m5
	move #FFTbuff,r0
	move #<WindowLen-1,m0
	.loop #WindowLen                ;apply input window
	  move X:(r1),x0 Y:(r5)+,y0     ;x0 = sample_I, y0 = input window
	  mpyr x0,y0,a Y:(r1)+,x0       ;a = x0*y0, x0 = sample_Q
	  mpyr x0,y0,a a,X:(r0)         ;a = x0*y0, save windowed I
	  move a,Y:(r0)+                ;save windowed Q into FFTbuff
	.endl
	move r1,X:<WindowInpTapPtr

	if SPY_RxWindow
	 jsr <SpySync
	 .if <cc>
	  .loop #WindowLen
	    move X:(r1)+,a
	    jsr <SpyA
	    nop
	  .endl
	  .loop #WindowLen
	    move Y:(r1)+,a
	    jsr <SpyA
	    nop
	  .endl
	 .endi
	endif

	jsr <DoFFT              ;execute FFT

      if RxAGC
       if SPY_PeakMS
	clr b X:<RxAudioPeak,a
      if !RxAGCsensePeak
	move b,X:<RxAudioPeak
      endif
	jsr <SpyA
	move X:<RxAudioMS,a
	jsr <SpyA
       endif

      if MonitorRxAGC
	UpLED clr
	DownLED clr
      endif

      move X:<RxCodecPtr,r2     ;set r2 to address the CODEC's buffer
      move #BufLen*4-1,m2       ;set m2 so r2 wraps in the CODEC's buffer
				;we will need r2/m2 for RxGainUp/Down
      if RxAGCsensePeak
	move #RxAGCmaxPeak,x0   ;check if peak is above the given max.
	move X:<RxAudioPeak,b
	cmp x0,b #<0,b
	move b,X:<RxAudioPeak   ;clear the peak
	jcc <GainDown           ;reduce gain if the peak is too high
      endif

        move X:<RxAudioMS,a
        move #RxAGCminMS,x0     ;see, if the MS is below the minimal level
	cmp x0,a
	jcc <CheckMSmax         ;if not, then check if above the max. level
				;if so, increase the CODEC's input gain
	move X:<RxAGCcount,a    ;decrement the timeout
	move #>1,x0
	sub x0,a 
	move a,X:<RxAGCcount
	jgt <CheckMS_OK         ;leave if not yet zero
	jsr <RxGainUp           ;increase the CODEC's gain
	jcs <CheckMS_OK
	move #0.7071,y0         ;increase the MS to follow
	move X:<RxAudioMS,x0    ;the gain change faster
	mpyr x0,y0,a #>RxAGCfall,x0 ;preset the AGC "timer" to prohibit
	asl a x0,X:<RxAGCcount      ;gain increase for the RxAGCfall period
	move a,X:<RxAudioMS
      if MonitorRxAGC
	UpLED set
      endif
	jmp <CheckMS_OK

CheckMSmax                      ;is the MS above the given maximum
	move #>RxAGChold,y0     ;preset the AGC "timer" to prohibit gain increase
	move #RxAGCmaxMS,x0     ;for the RxAGChold period
	cmp x0,a y0,X:<RxAGCcount ;MS above maximum ?
	jcs <CheckMS_OK         ;if not then we are done
GainDown
	jsr <RxGainDown         ;otherwise decrease the CODEC's gain
	jcs <CheckMS_OK
	move #0.7071,y0         ;decrease the MS to follow expected
	move X:<RxAudioMS,x0    ;gain reduction faster
	mpyr x0,y0,a #>RxAGChold,x0     ;initialize the AGC hold count-down
	move a,X:<RxAudioMS
	move x0,X:<RxAGCcount
      if MonitorRxAGC
	DownLED set
      endif

CheckMS_OK

      endif

	if SPY_RxFFT            ;SPY on the FFT result on the Rx signal
	 jsr <SpySync
	 .if <cc>
	  move #FFTbuff,r0
	  move #<0,m0
	  move #WindowLen/2,n0
	  .loop #WindowLen
	    move X:(r0),a
	    jsr <SpyA
	    move Y:(r0)+n0,a
	    jsr <SpyA
	    nop
	  .endl
	 .endi
	endif

        move X:<RxPipePtr,r4            ;copy the spectrum into the processing
        move #FFTprocLen*RxPipeLen-1,m4 ;pipe and reorder the FFT result
        move #FFTbuff,r0                ;properly on the way
        move #0,m0
        move #WindowLen/2,n0
      if FFTprocStart>0
        rep #FFTprocStart
          move (r0)+n0
      endif
        .loop #FFTprocLen
          move L:(r0)+n0,ab
          move ab,L:(r4)+
        .endl
;        move r4,X:<RxPipePtr

;        if SPY_RxCarFreq
;          move X:<RxCarFreq,a
;          jsr <SpyA
;        endif

;now we can/should do some processing of the FFT data

        move X:<RxPipePtr,r4            ;compute spectral power
        move #FFTprocLen*RxPipeLen-1,m4
        move #FFTpower,r0
        move #FFTprocLen-1,m0
        move X:(r4),x0        ;x0 = I
        .loop #FFTprocLen       
          mpy x0,x0,a Y:(r4)+,x0 ;square the I part
          mac x0,x0,a X:(r4),x0 ;add squared Q part, load next I
          move a,L:(r0)+        ;save the power
        .endl
        move r4,X:<RxPipePtr    ;move forward the pipe pointer

        move #FFTprocLen,n0
        move #NoiseFloorExclude,n1 ;8 highest bins we exclude
        jsr <PartialSort
        clr a L:(r0)+,x
        .loop n0
          add x,a L:(r0)+,x
        .endl
        rep #FFTprocLenLog
          asr a                 ;a = estimated noise floor for this FFT sample

;        move L:<NoiseFloor_0,b  ;filter the noise floor
;        move L:<NoiseFloor,x
;        add x,b
;        asr b
;        sub b,a L:<NoiseFloor_0,b
;        rep #NoiseFloorFollow-1
;          asr a
;        add a,b L:<NoiseFloor_0,a
;        sub x,a b,L:<NoiseFloor_0
;        rep #NoiseFloorFollow
;          asr a
;        add x,a
;        move a,L:<NoiseFloor

        move L:<NoiseFloor,b
        LongDecayAver a,b,#NoiseFloorFollow
        move b,L:<NoiseFloor

                                ;compute spectral powers, find the top one.
        clr b X:<RxPipePtr,r4   ;b = supposed top
        move #FFTprocLen*RxPipeLen-1,m4
        move #FFTpower,r0
        move #FFTprocLen-1,m0
        move X:(r4),x0        ;x0 = I
        .loop #FFTprocLen
          mpy x0,x0,a Y:(r4)+,x0 ;square the I part
          mac x0,x0,a X:(r4),x0 ;add squared Q part, load next I
          cmp b,a a,L:(r0)+     ;compare with peak (b), save the power
          tgt a,b               ;if higher than peak = this power
        .endl
                                ;b = the peak power of the spectrum to be
                                ;    processed.

        move L:<NoiseFloor,a            ;compute the threshold
        rep #Threshold
          asl a
        cmp b,a
        tlt b,a                 ;a = max(peak power, threshold)
                                        ;find the scaling factor (r1)
        tst a #<0,r1                    ;for the maximum power
        move #$FFFF,m1
        jle <TxPrepareIFFT
        .repeat
          asl a (r1)+
        .until <es>
        asr a

        move L:<NoiseFloor,a
        rep r1
          asl a
        asr a
        rep #Threshold
          asl a
        move a,y0                       ;y0 = the reference for noise cut
        move X:<RxPipePtr,r4
        move #FFTprocLen*RxPipeLen-1,m4
        move #FFTpower,r0
        move #FFTprocLen-1,m0
        move #SigToThres,r5
        move #FFTprocLen-1,m5
        .loop #FFTprocLen               ;loop over spectral bins
          move L:(r0)+,a                ;get the power for this bin
          rep r1                        ;scale it as we did with the reference
            asl a
          asr a
          cmp y0,a #$7FFFFF,x1          ;if greater than reference
          jge <Proc_next                ;then don't touch this bin
          andi #$FE,ccr                 ;if lower then compute:
          rep #24                       ;power/reference
            div y0,a                    ;and then the attenuation factor in y1
          move a0,x1
          if ParabolicSmooth
	    move a0,x0
	    mpy x0,x0,b x0,a
	    subl b,a
	    rnd a
	   if ParabolicSmooth>1
	    move a,x0
	    mpyr x0,x0,a
	   endif
	    move a,y1
	  else
	    move a0,y1
	  endif
          move X:(r4),x0                ;attenuate this bin L:(r4) by
          mpyr x0,y1,a Y:(r4),x0        ;by the computed factor (y1)
          mpyr x0,y1,a a,X:(r4)
          move a,Y:(r4)
Proc_next move (r4)+
          move x1,Y:(r5)+               ;save the signal-to-threshold
        .endl

      if SendSigToThres
        move L:<NoiseFloor,a
        jsr <Log2               ;compute the log-2 of the noise floor
        neg a #>2,b
        move a1,a0
        move a2,a1
        move #<0,a2
        add b,a
        rep #5
          asl a
        move a,L:<Scratch
        move #FFTpower+12,r1
        move #FFTprocLen-1,m1
        .loop #180
          move L:(r1)+,a
          jsr <Log2
          neg a L:<Scratch,b
          move a1,a0
          move a2,a1
          move #<0,a2
          rep #5
            asl a
          sub a,b
          rnd b #>$FF,x0
          cmp x0,b #>1,x1
          tgt x0,b
          cmp x1,b
          tlt x1,b
          move b,x0
          putc
          nop
        .endl
        move #<0,x0
        putc
      endif

      if 0
;transmit the signal-to-threshold on the serial port
        .loop #FFTprocLen       ;send the signal-to-threshold values
          move Y:(r5)+,a        ;out on serial port
          rep #9                ;one byte per FFT bin
            asl a               ;but avoid zeros as a zero is the record
          clr a a2,x0           ;terminator
          cmp x0,a #<1,a1       ;if zero is to be sent then make it
          .if <eq>              ;a minimal non-zero (=1)
            move a1,x0
          .endi
          putc
          nop
        .endl
        move #0,x0              ;send the terminator (NULL)
        putc
      endif

TxPrepareIFFT
        move #FFTbuff+FFTprocStart,r0   ;copy the spectral data
        move #WindowLen-1,m0            ;into the FFT work buffer
        move X:<RxPipePtr,r4
        move #FFTprocLen*RxPipeLen-1,m4
        .loop #FFTprocLen
          move L:(r4)+,ab
          neg b a,X:(r0)                ;invert the Q part so we effectively
          move b,Y:(r0)+                ;so the Inverse FFT
        .endl
      if (WindowLen-FFTprocLen)>0
        clr a
        .loop #WindowLen-FFTprocLen
          move a,L:(r0)+
        .endl
      endif

        if SPY_RxCarFreq
	  move X:<RxCarFreq,a
	  jsr <SpyA
	endif

TxDoFFT  jsr <DoFFT             ;execute inverse FFT

TxDoWindow                      ;overlap the new window
	move #FFTbuff,r0
	move #<0,m0
	move #WindowLen/2,n0
        move #TxWindow,r5
	move #<WindowLen-1,m5
	move X:<WindowOutTapPtr,r1 ;r1 to address the output window tap
	move #<WindowLen-1,m1
	move X:(r0),x0 Y:(r5)+,y0
        .loop #WindowLen-FFTstep
	  move L:(r1),ab
	  macr x0,y0,a Y:(r0)+n0,x0
	  macr x0,y0,b X:(r0),x0 Y:(r5)+,y0
	  move ab,L:(r1)+
	.endl
        .loop #FFTstep-1
	  mpyr x0,y0,a Y:(r0)+n0,x0
	  mpyr x0,y0,b X:(r0),x0 Y:(r5)+,y0
	  move ab,L:(r1)+
	.endl
	mpyr x0,y0,a Y:(r0)+n0,x0
	mpyr x0,y0,b a,X:(r1)
	move b,Y:(r1)+

	move X:<TxCodecPtr,r2
	move #BufLen*4-1,m2
        move #AliasFilterOutI_left,r4
	move #<AliasFilterLen-1,m4
        move #<AliasFilterOutQ_left-AliasFilterOutI_left,n4
        move #AliasFilterOutI_right,r5
        move #<AliasFilterLen-1,m5
        move #<AliasFilterOutQ_right-AliasFilterOutI_right,n5
        move (r2)+
        .loop #FFTstep          ;output interpolation
;          move #4*(AliasFilterLen-DecimateRatio),n2
          move #4*AliasFilterLen,n2
          move L:(r1)+,y       ;y1:y0 = processed samples (I/Q) ready to be output
          move (r2)-n2
          move #<3,n2
          .loop #DecimateRatio
            move (r2)+
            move (r2)+n2
;            move Y:(r2)+,a
;            asl a Y:(r2)-,b
;            asl b a,Y:(r2)+
;            move b,Y:(r2)+n2
          .endl
          move #<4,n2
          move r2,x1
          move Y:(r4+n4),x0
          .loop #AliasFilterLen-DecimateRatio   ;for the left channel
            move Y:(r2),a
            mac x0,y0,a Y:(r4)+,x0
            macr x0,y1,a Y:(r4+n4),x0
            move a,Y:(r2)+n2
          .endl
	  .loop #DecimateRatio
            mpy x0,y0,a Y:(r4)+,x0
            macr x0,y1,a Y:(r4+n4),x0
	    move a,Y:(r2)+n2
	  .endl
          move x1,r2
          move Y:(r5+n5),x0
          move (r2)+
          .loop #AliasFilterLen-DecimateRatio   ;for the right channel
            move Y:(r2),a
            mac x0,y0,a Y:(r5)+,x0
            macr x0,y1,a Y:(r5+n5),x0
            move a,Y:(r2)+n2
	  .endl
	  .loop #DecimateRatio
            mpy x0,y0,a Y:(r5)+,x0
            macr x0,y1,a Y:(r5+n5),x0
	    move a,Y:(r2)+n2
          .endl
          move (r2)-
        .endl
	move (r2)-
	move r2,X:<TxCodecPtr
	move r1,X:<WindowOutTapPtr

	jmp     <ProcessLoop

;------------------------------------------------------------------------------
;CODEC's buffer synchronization to the sample stream

CheckSampleBlock        ;a = how many samples we want
	asl a           ;r2 = where we wait for the block to start
	asl a #>2,x1
	add x1,a
	move a,x1
CheckSampleBlock_1
	move r7,a
	move r2,x0
	sub x0,a
	jpl <CheckSample_cmp
	  move #>BufLen*4,x0
	  add x0,a
CheckSample_cmp
	cmp x1,a
	rts             ;on output: a,x0,x1 are modified
			;If carry=0 => we are sure that:
			;1. there are at least A new input samples at X:(r2)
			;   to be read
			;2. there are at least A locations at Y:(r2) to
			;   be written


WaitSampleBlock         ;wait for a block of samples from the CODEC
			;on input: a = how many samples we want
			;          r2 = where the block starts
	jsr <CheckSampleBlock
	jcc <WaitSample_ret
WaitSample_loop
	wait            ;wait (at low power) until an interrupt comes
	jsr <CheckSampleBlock_1
	jcs <WaitSample_loop
WaitSample_ret
	rts             ;on output: a,x0,x1 are modified
			;We are sure that:
			;1. there are at least A new input samples at X:(r2)
			;   to be read
			;2. there are at least A locations at Y:(r2) to
			;   be written

;------------------------------------------------------------------------------
;Mathematical functions

PI      equ     3.14159265358979323846

;this routine computes a cosine/sine pair using the sine ROM
;with a second order (linear+quadrature) approximation between table points
IQ                              ;x0 = angle ( -1 = -PI, +1 = +PI)
;        ori #%00000100,omr      ;enable the sine ROM table
	move #>$80,x1   ;shift out 8 most significant bits
	mpy x0,x1,a  #>$FF,x0
	move x0,m0
	and x0,a     #>$100,x0
	or x0,a      #<$40,n0
	move a1,r0      ;put the 8 most significant bits into r0 with offset = $100
	move a0,y0      ;save the remaining bits in y0
	jclr #23,y0,SinTable_lev2
	  move (r0)+
SinTable_lev2
	move Y:(r0+n0),x0       ;x0 = coarse cosine
	move Y:(r0),x1          ;x1 = coarse sine
	mpyr x1,y0,a  #PI/256.0,y1
	tfr x0,a  a,x1
	macr -x1,y1,a           ;a = fine cosine
	mpyr x0,y0,b  Y:(r0),x1
;        andi #%11111011,omr     ;disable the sine ROM table
	tfr x1,b  b,x1
	macr x1,y1,b  #PI*PI/2.0/65536.0,y1  ;b = fine sine
	mpyr y0,y0,a  a,x0
	move a,y0
	mpyr y0,y1,a
	tfr x0,a  a,y1
	macr -x0,y1,a  b,x1     ;a = super fine cosine
	macr -x1,y1,b           ;b = super fine sine
	rts                     ;x,y are modified
				;r0,m0,n0 are modified
				;maximum error is about 0.7E-6
				;execution time 4+64+4 clock cycles
				;including "jsr <IQ_lev2" and "rts"

                        ;2-based logarythm function
Log2                    ;a2:1:0 = 56-bit argument (0.0,+256.0)
	tst a #<0,r0    ;r0 ready for finding the integer part
	jmi <Log2_Neg   ;if argument negative...
	jeq <Log2_Inf   ;if zero...
	jnr <Log2_Frac  ;if already normalized
Log2_Norm               ;normalize the argument to the range <0.5,1.0)
	rep #8
	  norm r0,a     ;r0 counts the number of shifts performed
	jnn <Log2_Norm
			;r0 = approx. integer result
Log2_Frac               ;a = argument scaled to range <0.5,1.0) = X
	clr a a,x0      ;x0 = X
	move r0,a2      ;now we compute the finer fractional approximation
	asr a x0,x1     ;a = approx. integer part
	bchg #23,x0     ;x0 = X-1
	add x0,a        ;a += 2*(X-1) => first, linear correction
	add x0,a
	bchg #22,x1     ;x1 = X-1/2
	mpyr x0,x1,b #0.371,x1 ;b = (X-1)*(X-1/2)
	sub b,a b,x0    ;a += (X-1)*(X-1/2)
	macr -x0,x1,a   ;a += 0.371*(X-1)*(X-1/2)
	rts             ;a = log2(input) <-48.0,+8.0) => second, parabolic corr.
			;b,x,r0 are modified
			;error RMS is 0.005, maximum error is 0.008
Log2_Neg                ;for zero and negative arguments we give the most
Log2_Inf clr a          ;negative result: a = -48.0
	move #<$E8,a2
	rts             ;execution time is about 80 clock cycles
			;but becomes larger for very small arguments

;------------------------------------------------------------------------------
;Routines to move the CODEC's input gain up or down

	if RxAGC

RxGainUp                        ;increase the CODEC's input gain (both channels)
	clr a #>$0F0F00,x0      ;r2/m2 should point to the CODEC's buffer
	move Y:(r2),a1          ;get the CODEC's input control word
	and x0,a                ;extract the gain bits
	cmp x0,a  #>$010100,x0  ;already maximum ?
	jeq <RxGainChange_abort ;if so then abort this attempt
	add x0,a  #>$F0F000,x0  ;if not, increment the gain by 1
	move a1,x1
	move Y:(r2),a1          ;and reload all the control words
	and x0,a  n2,x0         ;in the output buffer
	or x1,a  #<4,n2         ;make n2=4 for a moment
	.loop #BufLen
	  move a1,Y:(r2)+n2
	.endl
	move x0,n2              ;restore n2
	andi #$FE,ccr           ;clear carry
	rts                     ;modifies a,x0,x1

RxGainDown                      ;decrease the CODEC's input gain (both channels)
	clr a #>$0F0F00,x0      ;r2/m2 should point to the CODEC's buffer
	move Y:(r2),a1          ;get the CODEC's input control word
	and x0,a  #>$010100,x0  ;extract the gain bits
	sub x0,a  #>$F0F000,x0  ;attempt to decrease the gain
	jcs <RxGainChange_abort ;jump if overflow
	move a1,x1
	move Y:(r2),a1          ;reload all the input control words
	and x0,a  n2,x0         ;in the buffer with the new input gain
	or x1,a  #<4,n2         ;n2=4 for a moment
	.loop #BufLen
	  move a1,Y:(r2)+n2
	.endl
	move x0,n2              ;restore n2
	andi #$FE,ccr           ;clear carry
	rts                     ;modifies a,x0,x1

RxGainChange_abort
	ori #$01,ccr            ;set carry to indicate that we couldn't
	rts                     ;change the gain

	endif

;------------------------------------------------------------------------------

        if SPY

SpySync move a10,L:<SpySave     ;output: carry=1 => no spy request
	move a2,X:<SpySave+1    ;carry=0 => spy request !
	move x0,Y:<SpySave+1    ;512 words (jsr <SpyA) should follow
	move x1,Y:<SpyCount
	move X:<SpyCount,a
	tst a
	jne <SpyCont
	lookc 0
	jcs <Spy_end
	move #>'S',a
	cmp x0,a
	ori #$01,ccr
	jne <Spy_end
	move #>'P',x0
	putc
	move #>512,a
	move a,X:<SpyCount
SpyCont andi #$FE,ccr
	jmp <Spy_end

SpyA    move a10,L:<SpySave
	move a2,X:<SpySave+1
	move x0,Y:<SpySave+1
	move x1,Y:<SpyCount
	move X:<SpyCount,a
	tst a
	jne <Spy_copy

Spy_check
	lookc 0
	jcs <Spy_end
	move #>'S',a
	cmp x0,a
	jne <Spy_end
	move #>'P',x0
	putc
	move #>512,a
Spy_copy
	move #>1,x0
	sub x0,a
	move a,X:<SpyCount

	move X:<SpySave,a
	rep #8
	  lsr a
	move a1,x0
	putc
	move X:<SpySave,a
	rep #16
	  lsr a
	move a1,x0
	putc

Spy_end move L:<SpySave,a10
	move X:<SpySave+1,a2
	move Y:<SpySave+1,x0
	move Y:<SpyCount,x1
	rts

	endif

;------------------------------------------------------------------------------

        ;Partial sort to get out N largest powers
PartialSort     ;L:(r0)/m0 = data, n0 = data size, n1 = N (N>1 !)
        move #$FFFF,m1
        .loop n1
          move n0,r1
          move L:(r0)+,a        ;first element as supposed maximum
          move (r1)-            ;decrement n0
          move r1,n0
          move r0,r1            ;r1 = supposed maximum place (+1)
          .loop n0              ;look through the data and find the max. one
            move L:(r0)+,b
            cmp a,b
            tgt b,a r0,r1
          .endl                 ;at the end of this loop a=max, r1=pos+1
          move (r0)-n0
          move L:-(r0),b        ;b = first element
          move b,L:-(r1)        ;swap with the max. so max. is first
          move a,L:(r0)+        ;r0 points at the first non-sorted element
        .endl
        rts     ;L:(r0)/m0 = remining data, n0 = remining size
                ;r1,m1,a,b are modified

;------------------------------------------------------------------------------

Initialize      ;initialize registers, buffers, windows, etc.

        ori #%00000100,omr      ;enable the sine ROM table
                                ;this is for the IQ routine

        clr a #WindowInpTap,r0  ;clean the window taps
        move #<WindowLen-1,m0
        rep #WindowLen
          move a,L:(r0)+
        move #WindowOutTap,r0
        rep #WindowLen
          move a,L:(r0)+

        clr a #RxPipe,r0        ;clear the receiver pipe
        move #WindowLen*RxPipeLen-1,m0
        rep #WindowLen*RxPipeLen
          move a,L:(r0)+

      if WindowLen<=256
        move #$180,r0           ;initialize the sine/cosine table
        move #<$FF,m0           ;by taking every Nth value from the on-chip
        move #<$100/WindowLen,n0 ;ROM sine table (N=256/WindowLen)
        move #FFTcoef,r4         ;This works only for FFT lengths up to 256
        move #<WindowLen-1,m4     ;note that the table for the FFT starts
        move #<3*(WindowLen/4),n4 ;with cos=-1.0 and sin=0.0
        .loop #WindowLen          ;X memory contains the cosine
          move Y:(r0)+n0,a        ;Y memory contains the sine
          move a,X:(r4+n4)
          move a,Y:(r4)+
        .endl
      else
        move #FFTcoef,r4        ;initialize the coeff. table for the FFT
        move #WindowLen-1,m4
        move #-1.0,a            ;we start at phase = -PI
        move #2.0/WindowLen,b   ;and we step by 2*PI/WindowLen
        move ab,L:<Scratch      ;a = phase, b = step
        .loop WindowLen         ;note that the table for the FFT starts
          move a,x0             ;with cos=-1.0 and sin=0.0
          jsr <IQ               ;X memory contains the cosine
          move ab,L:(r4)+       ;Y memory contains the sine
          move L:<Scratch,ab
          add b,a
          move a,X:<Scratch
        .endl
      endif

        move #RxWindow,r1       ;here we compute the FFT window
        move #WindowLen-1,m1
        move #TxWindow,r2
        move m1,m2
        clr a #1.0/WindowLen,b  ;we start at phase = 0
        move ab,L:<Scratch      ;and we step by PI/WindowLen
        .loop #WindowLen        ;so we end up at PI
           move a,x0
           jsr <IQ              ;we compute the SIN(phase)
        if FinerShaping         ;and optionally we take SIN^2
           move b,x0
           mpyr x0,x0,b
        endif
           move b,Y:(r2)+       ;save for the output window
           rep #WindowLenLog-1  ;for the input we scale down
             asr b              ;the window to avoid overflows
           move b,Y:(r1)+
           move L:<Scratch,ab
           add b,a
           move a,X:<Scratch
        .endl

;        bclr #12,X:<<$FFF0    ;disable SCI Tx interrupts
;        bclr #11,X:<<$FFF0    ;disable SCI Rx interrupts
      if NewBaudRate        ;set new baud rate (copied from LEONID)
;        jclr #0,X:<<$FFF1,*  ;wait until all data is sent out
        movep #(xtal+2*16*NewBaudRate)/(2*2*16*NewBaudRate)-1,X:<<$FFF2
      endif

        move #Buffer+2,r7        ;for the CODEC's interrupt routine
        move #BufLen*4-1,m7

        move #Buffer,r2          ;for "cdctrl" to initialize the buffer
        move #<4-1,n2
        move #BufLen*4-1,m2
                        ;initialize input/output control words in the buffer
                        ;zero input/output data
      if (EVM56K>0)     ;for EVM56002 use MIC input
        ctrlcd  1,r2,BufLen,MIC,RxGain,RxGain,LINEO|HEADP,TxAtten,TxAtten
      else              ;for DSPCARD4 use LINE input
        ctrlcd  1,r2,BufLen,LINEI,RxGain,RxGain,LINEO|HEADP,TxAtten,TxAtten
      endif
        opencd SampleFreq/1000.0,HPF     ;start taking samples at given rate

        jmp <ProcessLoop

;*************************************************************************
;Internal data RAM

	LOMEM X:$0000,Y:$0000,L:$0000
	HIMEM X:$00FF,Y:$00FF,L:$00FF

	org L:user_data

	if SPY
SpySave dc 0,0
SpyCount dc 0
	endif

Scratch ds 1    ;general purpose scratch variable

PowerPeak dc 0  ;the peak power in the spectrum

NoiseFloor   dc 0
;NoiseFloor_0 dc 0

LastL = *
	org X:LastL
	org Y:LastL

	org X:
WindowInpTapPtr dc WindowInpTap
WindowOutTapPtr dc WindowOutTap
RxCodecPtr    dc Buffer
TxCodecPtr    dc Buffer

;RxCarPhase    dc 0      ;Rx mixer carrier phase and initial frequency
;RxCarFreq     dc 0

RxPipePtr   dc RxPipe

	if RxAGC
RxAudioPeak     dc 0            ;peak level of the input audio
RxAudioMS       dc 0            ;Mean Square (the power) of the input audio
RxAGCcount      dc RxAGCfall
	endif

PowerPeakCount dc PowerPeakHold

;IntrTmp ds 1            ;temporary storage for saving a register
			;during an interrupt
	org Y:

	org X:
LastX = *
	org Y:
LastY = *
	if @cvi(LastX)>=@cvi(LastY)
	  org L:LastX
	else
	  org L:LastY
	endif

      if FFTprocLen<=128
        org L:
FFTpower dsm FFTprocLen
      endif

;*************************************************************************
;External L (X/Y) data RAM

	if (EVM56K>0)
	  LOMEM X:$2000,Y:$2000,L:$2000
	  HIMEM X:$3FFF,Y:$3FFF,L:$3FFF
	else
	  LOMEM X:$0200,Y:$0200,L:$0200
	  HIMEM X:$1FFF,Y:$1FFF,L:$1FFF
	endif

      if (EVM56K>0)
	org L:$2000
      else
	org L:$200
      endif

	org L:
WindowInpTap dsm WindowLen
WindowOutTap dsm WindowLen
FFTcoef dsm WindowLen   ;sine/cosine table for the FFT and other stuff
FFTbuff dsm WindowLen   ;Work buffer for the FFT routine
      if FFTprocLen>128
FFTpower = FFTbuff      ;and usefull for temporary storage of spectal powers
      endif

RxPipe  dsm FFTprocLen*RxPipeLen        ;here we store and process
                                        ;the spectral data
LastL = *
	org X:LastL
	org Y:LastL

	org X:
LastX = *
	org Y:
LastY = *
	if @cvi(LastX)>=@cvi(LastY)
	  org L:LastX
	else
	  org L:LastY
	endif

Buffer  dsm BufLen*4    ;CODEC's input/output buffer

;*************************************************************************
;external Y-only data RAM

	if (EVM56K>0)
          LOMEM Y:$0200
          HIMEM Y:$1FFF
	else
	  LOMEM Y:$2000
	  HIMEM Y:$3FFF
	endif

      if (EVM56K>0)
        org Y:$0200
      else
	org Y:$2000
      endif

        org Y:
AliasFilterInpI dsm AliasFilterLen      ;anti-alias input filter (I/Q)
AliasFilterInpQ dsm AliasFilterLen

        org Y:
AliasFilterOutI_left  dsm AliasFilterLen      ;anti-alias output filter (I/Q)
AliasFilterOutQ_left  dsm AliasFilterLen      ;for the LEFT ear
AliasFilterOutI_right dsm AliasFilterLen      ;anti-alias output filter (I/Q)
AliasFilterOutQ_right dsm AliasFilterLen      ;for the RIGHT ear

        org Y:
RxWindow   dsm WindowLen           ;input FFT window
TxWindow   dsm WindowLen           ;output FFT window

        org Y:
SigToThres dsm FFTprocLen          ;signal to threshold ratio

;*************************************************************************
;constant tables: FIR and window shapes, FFT coefficiants, etc.

	if (EVM56K>0)
	  LOMEM X:$2000,Y:$0200,L:$2000
	  HIMEM X:$3FFF,Y:$3FFF,L:$3FFF
	else
	  LOMEM X:$0200,Y:$0200,L:$0200
	  HIMEM X:$1FFF,Y:$3FFF,L:$1FFF
	endif

;w0      equ     0.35875 ;coeff. for Blackman-Harris 4 term
;w1      equ     0.48829 ;minimum sidelobe window (copied from Motorola BBS)
;w2      equ     0.14128
;w3      equ     0.01168

;We use a simpler window for the anti-alias filter
w0      equ 0.5
w1      equ 0.5
w2      equ 0.0
w3      equ 0.0

LowFreq = (AliasCenter-AliasWidth/2)/SampleFreq
UppFreq = (AliasCenter+AliasWidth/2)/SampleFreq

        org Y:AliasFilterInpI

time = -@cvf(AliasFilterLen/2)+0.5
count   set 0
        dup AliasFilterLen
angle = PI*time/@cvf(AliasFilterLen/2)
Window = w0+w1*@cos(angle)+w2*@cos(2.0*angle)+w3*@cos(3.0*angle)
Filter = (@sin(2.0*PI*time*UppFreq)-@sin(2.0*PI*time*LowFreq))/(PI*time)
time = time+1.0
        dc  Window*Filter
count   set count+1
        endm

        org Y:AliasFilterInpQ

time = -@cvf(AliasFilterLen/2)+0.5
count   set 0
        dup AliasFilterLen
angle = PI*time/@cvf(AliasFilterLen/2)
Window = w0+w1*@cos(angle)+w2*@cos(2.0*angle)+w3*@cos(3.0*angle)
Filter = (-@cos(2.0*PI*time*UppFreq)+@cos(2.0*PI*time*LowFreq))/(PI*time)
time = time+1.0
        dc  Window*Filter
count   set count+1
        endm

;We keep same Low/UppFreq for the output filters
;LowFreq = (AliasCenter-AliasWidth/2)/SampleFreq
;UppFreq = AliasCenter/SampleFreq

        org Y:AliasFilterOutI_left

time = -@cvf(AliasFilterLen/2)+0.5
count   set 0
	dup AliasFilterLen
angle = PI*time/@cvf(AliasFilterLen/2)
Window = w0+w1*@cos(angle)+w2*@cos(2.0*angle)+w3*@cos(3.0*angle)
Filter = (@sin(2.0*PI*time*UppFreq)-@sin(2.0*PI*time*LowFreq))/(PI*time)
time = time+1.0
	dc  Window*Filter
count   set count+1
	endm

        org Y:AliasFilterOutQ_left

time = -@cvf(AliasFilterLen/2)+0.5
count   set 0
	dup AliasFilterLen
angle = PI*time/@cvf(AliasFilterLen/2)
Window = w0+w1*@cos(angle)+w2*@cos(2.0*angle)+w3*@cos(3.0*angle)
Filter = (-@cos(2.0*PI*time*UppFreq)+@cos(2.0*PI*time*LowFreq))/(PI*time)
time = time+1.0
	dc  Window*Filter
count   set count+1
	endm

;LowFreq = AliasCenter/SampleFreq
;UppFreq = (AliasCenter+AliasWidth/2)/SampleFreq

        org Y:AliasFilterOutI_right

time = -@cvf(AliasFilterLen/2)+0.5
count   set 0
	dup AliasFilterLen
angle = PI*time/@cvf(AliasFilterLen/2)
Window = w0+w1*@cos(angle)+w2*@cos(2.0*angle)+w3*@cos(3.0*angle)
Filter = (@sin(2.0*PI*time*UppFreq)-@sin(2.0*PI*time*LowFreq))/(PI*time)
time = time+1.0
	dc  Window*Filter
count   set count+1
	endm

        org Y:AliasFilterOutQ_right

time = -@cvf(AliasFilterLen/2)+0.5
count   set 0
	dup AliasFilterLen
angle = PI*time/@cvf(AliasFilterLen/2)
Window = w0+w1*@cos(angle)+w2*@cos(2.0*angle)+w3*@cos(3.0*angle)
Filter = (-@cos(2.0*PI*time*UppFreq)+@cos(2.0*PI*time*LowFreq))/(PI*time)
time = time+1.0
	dc  Window*Filter
count   set count+1
	endm

;*************************************************************************

	end

